<?php
// Enable error reporting for debugging (DISABLE IN PRODUCTION)
session_start();
require_once 'conn.php';

// === DEBUG START ===
file_put_contents('debug_transfer.txt', 
    "=== NEW REQUEST ===\n" .
    "Time: " . date('Y-m-d H:i:s') . "\n" .
    "Method: " . $_SERVER['REQUEST_METHOD'] . "\n" .
    "POST Data: " . print_r($_POST, true) . "\n" .
    "Session ID: " . ($_SESSION['id'] ?? 'NOT SET') . "\n" .
    "===================\n\n",
    FILE_APPEND
);

// Secure log function
function logError($message, $level = 'INFO') {
    $logFile = __DIR__ . '/transfer_log.txt';
    $timestamp = date('Y-m-d H:i:s');
    $sanitized = preg_replace('/\b\d{3,4}\b/', '****', $message); // Hide PINs in logs
    file_put_contents($logFile, "[$timestamp][$level] $sanitized\n", FILE_APPEND);
}

// CSRF Token Validation
function validateCSRF($token) {
    if (!isset($_SESSION['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $token)) {
        return false;
    }
    return true;
}

// Rate Limiting (Simple implementation)
function checkRateLimit($user_id) {
    $limit_key = 'transfer_limit_' . $user_id;
    $limit_time = 'transfer_time_' . $user_id;
    
    if (!isset($_SESSION[$limit_key])) {
        $_SESSION[$limit_key] = 1;
        $_SESSION[$limit_time] = time();
        return true;
    }
    
    // Reset after 1 hour
    if (time() - $_SESSION[$limit_time] > 3600) {
        $_SESSION[$limit_key] = 1;
        $_SESSION[$limit_time] = time();
        return true;
    }
    
    // Max 10 transfers per hour
    if ($_SESSION[$limit_key] >= 10) {
        return false;
    }
    
    $_SESSION[$limit_key]++;
    return true;
}

// Check if user is logged in
if (!isset($_SESSION['id'])) {
    logError("Unauthorized access attempt", 'WARNING');
    header("Location: ../auth-login.php?error=1&message=Please login first");
    exit();
}

if (isset($_POST['transfer'])) {
    try {
        logError("=== Transfer Started ===");
        logError("POST data received: " . print_r(array_keys($_POST), true));
        
        // Validate CSRF Token
        if (!isset($_POST['csrf_token'])) {
            logError("CSRF token missing from POST data", 'SECURITY');
            header("Location: ../transfer.php?error=1&message=Security token missing. Please refresh and try again.");
            exit();
        }
        
        if (!validateCSRF($_POST['csrf_token'])) {
            logError("CSRF token validation failed", 'SECURITY');
            header("Location: ../transfer.php?error=1&message=Invalid security token. Please refresh and try again.");
            exit();
        }
        
        $sender_id = intval($_SESSION['id']);
        
        // Check rate limit
        if (!checkRateLimit($sender_id)) {
            logError("Rate limit exceeded for user: $sender_id", 'WARNING');
            header("Location: ../transfer.php?error=1&message=Too many transfers. Please try again later.");
            exit();
        }
        
        // Sanitize and validate inputs
        $recipient_email = filter_var($_POST['recipient_email'], FILTER_SANITIZE_EMAIL);
        if (!filter_var($recipient_email, FILTER_VALIDATE_EMAIL)) {
            logError("Invalid email format: $recipient_email", 'ERROR');
            header("Location: ../transfer.php?error=1&message=Invalid email address format.");
            exit();
        }
        
        $amount = floatval($_POST['amount']);
        $description = substr(trim($_POST['description'] ?? ''), 0, 200);
        
        logError("Sender ID: $sender_id");
        logError("Recipient Email: $recipient_email");
        logError("Amount: $amount");
        
        // Validate amount
        if ($amount <= 0 || $amount > 1000000) {
            logError("Invalid amount: $amount", 'ERROR');
            header("Location: ../transfer.php?error=1&message=Invalid amount. Must be between $0.01 and $1,000,000.");
            exit();
        }
        
        // Start transaction BEFORE any queries
        $conn->begin_transaction();
        
        try {
            // Get sender details with row lock to prevent race conditions
            $stmt = $conn->prepare("SELECT id, balance, pin, fullname, email FROM users WHERE id = ? FOR UPDATE");
            $stmt->bind_param("i", $sender_id);
            $stmt->execute();
            $sender_result = $stmt->get_result();
            
            if ($sender_result->num_rows == 0) {
                throw new Exception("SENDER_NOT_FOUND");
            }
            
            $sender = $sender_result->fetch_assoc();
            $stmt->close();
            
            logError("Sender Balance: " . $sender['balance']);
            
            // Check if sender has sufficient balance
            if ($sender['balance'] < $amount) {
                throw new Exception("INSUFFICIENT_BALANCE");
            }
            
            // Get recipient details with row lock
            $stmt = $conn->prepare("SELECT id, balance, fullname, email FROM users WHERE email = ? FOR UPDATE");
            $stmt->bind_param("s", $recipient_email);
            $stmt->execute();
            $recipient_result = $stmt->get_result();
            
            if ($recipient_result->num_rows == 0) {
                throw new Exception("RECIPIENT_NOT_FOUND");
            }
            
            $recipient = $recipient_result->fetch_assoc();
            $recipient_id = $recipient['id'];
            $stmt->close();
            
            logError("Recipient ID: $recipient_id - " . $recipient['fullname']);
            
            // Check if trying to transfer to self
            if ($sender_id == $recipient_id) {
                throw new Exception("SELF_TRANSFER");
            }
            
            // Verify PIN if set
            if (!empty($sender['pin'])) {
                if (empty($pin)) {
                    throw new Exception("PIN_REQUIRED");
                }
                
                // Use password_verify if PINs are hashed, or hash_equals for comparison
                if (!hash_equals($sender['pin'], $pin)) {
                    logError("Incorrect PIN attempt for user: $sender_id", 'SECURITY');
                    throw new Exception("INCORRECT_PIN");
                }
                logError("PIN verified successfully");
            }
            
            // Calculate new balances
            $new_sender_balance = $sender['balance'] - $amount;
            $new_recipient_balance = $recipient['balance'] + $amount;
            
            // Update sender balance
            $stmt = $conn->prepare("UPDATE users SET balance = ? WHERE id = ?");
            $stmt->bind_param("di", $new_sender_balance, $sender_id);
            if (!$stmt->execute()) {
                throw new Exception("UPDATE_SENDER_FAILED");
            }
            $stmt->close();
            logError("Sender balance updated: $new_sender_balance");
            
            // Update recipient balance
            $stmt = $conn->prepare("UPDATE users SET balance = ? WHERE id = ?");
            $stmt->bind_param("di", $new_recipient_balance, $recipient_id);
            if (!$stmt->execute()) {
                throw new Exception("UPDATE_RECIPIENT_FAILED");
            }
            $stmt->close();
            logError("Recipient balance updated: $new_recipient_balance");
            
            // Generate unique transaction ID
            $transaction_id = 'TRF' . date('Ymd') . bin2hex(random_bytes(4));
            $current_date = date('Y-m-d H:i:s');
            
            logError("Transaction ID: $transaction_id");
            
            // Create transfers table if it doesn't exist
            $create_table = "CREATE TABLE IF NOT EXISTS transfers (
                id INT AUTO_INCREMENT PRIMARY KEY,
                transaction_id VARCHAR(50) UNIQUE,
                sender_id INT NOT NULL,
                recipient_id INT NOT NULL,
                amount DECIMAL(15,2) NOT NULL,
                description TEXT,
                status VARCHAR(20) DEFAULT 'completed',
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                INDEX idx_sender (sender_id),
                INDEX idx_recipient (recipient_id),
                INDEX idx_transaction (transaction_id)
            )";
            
            $conn->query($create_table);
            
            // Record transfer
            $stmt = $conn->prepare("INSERT INTO transfers (transaction_id, sender_id, recipient_id, amount, description, status, created_at) 
                                   VALUES (?, ?, ?, ?, ?, 'completed', ?)");
            $stmt->bind_param("siidss", $transaction_id, $sender_id, $recipient_id, $amount, $description, $current_date);
            if (!$stmt->execute()) {
                throw new Exception("TRANSFER_RECORD_FAILED");
            }
            $stmt->close();
            logError("Transfer record created");
            
            // Record as withdrawal for sender (if table exists)
            $stmt = $conn->prepare("INSERT INTO withdrawals (registration_id, amount, coin, transaction_id, status, date) 
                                   VALUES (?, ?, 'Transfer Out', ?, 'approved', ?)");
            if ($stmt) {
                $stmt->bind_param("idss", $sender_id, $amount, $transaction_id, $current_date);
                $stmt->execute();
                $stmt->close();
                logError("Withdrawal record created");
            }
            
            // Record as deposit for recipient (if table exists)
            $stmt = $conn->prepare("INSERT INTO deposits (registration_id, amount, coin, transaction_id, status, date) 
                                   VALUES (?, ?, 'Transfer In', ?, 'approved', ?)");
            if ($stmt) {
                $stmt->bind_param("idss", $recipient_id, $amount, $transaction_id, $current_date);
                $stmt->execute();
                $stmt->close();
                logError("Deposit record created");
            }
            
            // Commit transaction
            $conn->commit();
            logError("Transaction committed successfully");
            logError("=== Transfer Completed Successfully ===");
            
            // Success message
            $success_msg = urlencode("Transfer of $" . number_format($amount, 2) . " to " . htmlspecialchars($recipient['fullname']) . " completed successfully!");
            header("Location: ../transfer.php?message=$success_msg");
            exit();
            
        } catch (Exception $e) {
            // Rollback on error
            $conn->rollback();
            $error_code = $e->getMessage();
            logError("Transaction rolled back - $error_code", 'ERROR');
            
            // User-friendly error messages
            $error_messages = [
                'SENDER_NOT_FOUND' => 'Your account was not found.',
                'INSUFFICIENT_BALANCE' => 'Insufficient balance. Your balance is $' . number_format($sender['balance'] ?? 0, 2),
                'RECIPIENT_NOT_FOUND' => 'Recipient email not found. Please check the email address.',
                'SELF_TRANSFER' => 'You cannot transfer money to yourself.',
                'PIN_REQUIRED' => 'Please enter your PIN to complete the transfer.',
                'INCORRECT_PIN' => 'Incorrect PIN. Please try again.',
                'UPDATE_SENDER_FAILED' => 'Failed to process transfer. Please try again.',
                'UPDATE_RECIPIENT_FAILED' => 'Failed to process transfer. Please try again.',
                'TRANSFER_RECORD_FAILED' => 'Failed to record transfer. Please contact support.'
            ];
            
            $user_message = $error_messages[$error_code] ?? 'An unexpected error occurred. Please try again.';
            header("Location: ../transfer.php?error=1&message=" . urlencode($user_message));
            exit();
        }
        
    } catch (Exception $e) {
        logError("FATAL ERROR: " . $e->getMessage(), 'CRITICAL');
        header("Location: ../transfer.php?error=1&message=An unexpected error occurred. Please contact support.");
        exit();
    }
    
} else {
    logError("Invalid request method", 'WARNING');
    header("Location: ../transfer.php?error=1&message=Invalid request. Please try again.");
    exit();
}
?>